#!/bin/bash

echo "🚀 Starting Excel export process..."

# Input Params
ENVIRONMENT=${1:-development}
EXPORT_PATH=$2
MENU_ITEM_ID=$3
CHILD_PROCESS_SLUG=$4

# Input Validation
if [ -z "$ENVIRONMENT" ] || [ -z "$EXPORT_PATH" ] || [ -z "$MENU_ITEM_ID" ] || [ -z "$CHILD_PROCESS_SLUG" ]; then
  echo "❌ Missing required arguments. Usage:"
  echo "   bash export_excel.sh <env> <file_path> <menu_item_id> <child_process_slug>"
  exit 1
fi

# Environment File
ENV_FILE=".env.${ENVIRONMENT}"
if [ ! -f "$ENV_FILE" ]; then
  echo "❌ Environment file '$ENV_FILE' not found!"
  exit 1
fi

echo "✅ Using environment file: $ENV_FILE"

# Normalize line endings
sed -i 's/\r$//' "$ENV_FILE"

# Load environment vars safely (handle 'key = value' or 'key= 'value' ')
TEMP_ENV_FILE="/tmp/tmp_env_$RANDOM.env"
grep -v '^#' "$ENV_FILE" | sed -E "s/^[[:space:]]*([A-Za-z0-9_]+)[[:space:]]*=[[:space:]]*'?([^']*)'?[[:space:]]*$/\1=\"\2\"/" > "$TEMP_ENV_FILE"

set -a
source "$TEMP_ENV_FILE"
set +a
rm -f "$TEMP_ENV_FILE"

# Validate DB environment
if [ -z "$DATABASE_HOST" ] || [ -z "$DATABASE_PORT" ] || [ -z "$DATABASE_USER" ] || [ -z "$DATABASE_PASSWORD" ] || [ -z "$DATABASE_NAME" ]; then
  echo "❌ One or more database environment variables are missing."
  exit 1
fi

# Compose full DATABASE_URL
DATABASE_URL="postgresql://$DATABASE_USER:$DATABASE_PASSWORD@$DATABASE_HOST:$DATABASE_PORT/$DATABASE_NAME"
echo "📌 DATABASE_URL: $DATABASE_URL"

# Create export folder
mkdir -p "$EXPORT_PATH"

# Unique file name
TIMESTAMP=$(date +"%Y%m%d_%H%M%S")
UUID=$(date +%s%N | md5sum | head -c 12)
FILENAME="${UUID}_${TIMESTAMP}.xlsx"
FULL_PATH="$EXPORT_PATH/$FILENAME"

# API URL
API_URL="${API_BASE_URL:-http://localhost:3136}/api/common/grid-data-export-unauth"

echo "📡 Downloading Excel for menu_item_id=$MENU_ITEM_ID"
curl -s -X POST "$API_URL" \
  -H "Content-Type: application/json" \
  --data "{\"menu_item_id\": $MENU_ITEM_ID}" \
  --output "$FULL_PATH"

if [ $? -ne 0 ] || [ ! -f "$FULL_PATH" ]; then
  echo "❌ Failed to download Excel file."
  exit 1
fi

echo "✅ File saved to: $FULL_PATH"

# Update in database
echo "📝 Updating child_processes.downloadables in DB..."

PGPASSWORD=$DATABASE_PASSWORD psql -h "$DATABASE_HOST" -U "$DATABASE_USER" -d "$DATABASE_NAME" -p "$DATABASE_PORT" <<EOF
UPDATE child_processes
SET downloadables = '$FULL_PATH'
WHERE slug = '$CHILD_PROCESS_SLUG';
EOF

if [ $? -eq 0 ]; then
  echo "✅ child_processes.downloadables updated for slug: $CHILD_PROCESS_SLUG"
else
  echo "❌ Failed to update child_processes table."
fi
