import { PrismaClient, Prisma } from '@prisma/pg/client';

const prisma = new PrismaClient();

export async function userRolesSeeds(companyId: number) {
    try {
        const userRoleMapping = {
            "mukesh": ['Manager'],
            "raj": ['Supervisor'],
            "mukeshmtechcedence": ['Support Agent', 'Supervisor', 'Manager', 'Admin'],
            "mukeshchandran": ['Manager', 'Customer']
        };

        const roles = await prisma.role.findMany({
            where: {
                company_id: companyId,
                name: { in: Object.values(userRoleMapping).flat() }
            },
            select: { id: true, name: true },
        });

        const roleNameToIdMap = roles.reduce((map, role) => {
            map[role.name] = role.id;
            return map;
        }, {} as Record<string, number>);

        let superAdminOrAdminUser = await prisma.user.findFirst({
            where: {
                company_id: companyId,
                OR: [
                    { role: 'super_admin' },
                    { role: 'admin' }
                ]
            }
        });

        if (!superAdminOrAdminUser) {
            console.log(`User role seeder failed: Admin user not available for company id ${companyId}`);
            return false;
        }

        for (const [username, roleNames] of Object.entries(userRoleMapping)) {
            for (const roleName of roleNames) {
                const roleId = roleNameToIdMap[roleName];
                const user = await prisma.user.findUnique({
                    where: {
                        company_id_username: {
                            company_id: companyId,
                            username
                        }
                    }
                });

                if (!user) {
                    console.error(`Invalid user name ${roleName} for user ID ${username}`);
                    continue;
                }

                const userId = user.id;

                if (!roleId) {
                    console.error(`Invalid role name ${roleName} for user ID ${username}`);
                    continue;
                }

                const existingRole = await prisma.userRoles.findUnique({
                    where: {
                        company_id_user_id_role_id: {
                            company_id: companyId,
                            user_id: Number(userId),
                            role_id: roleId
                        }
                    }
                });

                if(!existingRole) {
                    await prisma.userRoles.create({
                        data: {
                            company_id: companyId,
                            user_id: Number(userId),
                            role_id: roleId
                        }
                    });
                    console.log(`Created new role permission for company ${companyId}, Role ${roleId}, UserID ${userId}`);
                }
            }
        }
    } catch (error) {
        console.error(`Error in userRolesSeeds: ${error}`);
        if (error instanceof Prisma.PrismaClientKnownRequestError && error.code === 'P2003' && error.meta) {
            console.error(`Foreign key constraint failed: ${error.meta.field_name}`);
        }
        throw error;
    } finally {
        // await prisma.$disconnect();
    }
}
