import { PrismaClient } from '@prisma/sql/client';
import { v4 as uuidv4 } from 'uuid';

const prisma = new PrismaClient();

export async function roleSeeds(company_id: number) {
  const roles = ['Admin', 'Manager', 'Supervisor', 'Customer', 'Support Agent'];

  try {
    console.log("🚀 Starting Role Seeder...");

    const user = await prisma.user.findFirst({
      where: { company_id, role: 'super_admin' },
    });

    if (!user) {
      console.log(`❌ Role seeder failed: No super_admin user for company_id ${company_id}`);
      return false;
    }

    const createdBy = user.id;
    const utcNow = new Date();

    // Fetch existing roles
    const existingRoles = new Set(
      (await prisma.role.findMany({
        where: { company_id, name: { in: roles } },
        select: { name: true },
      })).map(role => role.name)
    );

    const rolesToCreate = roles
      .filter(roleName => !existingRoles.has(roleName))
      .map(roleName => prisma.role.create({
        data: {
          uuid: uuidv4(),
          name: roleName,
          company_id,
          created_by: createdBy,
          created_at: utcNow,
        },
      }).then(role => {
        console.log(`✅ Role ${role.name} created successfully.`);
      }).catch(error => {
        console.error(`❌ Failed to create role ${roleName}:`, error);
      }))
    
    await Promise.all(rolesToCreate);

    console.log("✅ Role Seeder completed successfully.");
  } catch (error) {
    console.error("❌ Role seeding failed:", error);
  } finally {
    await prisma.$disconnect();
  }
}
