import { PrismaClient } from '@prisma/sql/client'
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

const action_types: any = {
  view: 'view',
  add: 'add',
  edit: 'edit',
  delete: 'delete',
  details: 'details',
  export_excel: 'export_excel',
  import: 'import',
  assign: 'assign',
};

async function getMasterEntityIdBySlug(slug: string, company_id: number) {
  const masterEntity = await prisma.masterEntity.findFirst({
    where: {
      company_id,
      entity_name: slug,
    },
  });
  return masterEntity ? masterEntity.id : null;
}

async function findOrCreatePermission(permissionData: any, company_id: number) {
  try {
    let { rights, ...roleData } = permissionData;

    let order_no = 1;
    for (const right of rights) {
      const slug = action_types[right] + '_' + roleData.slug;

      const masterEntityId = await getMasterEntityIdBySlug(
        roleData.slug,
        company_id
      );

      let existingPermission = await prisma.permissions.findFirst({
        where: {
          company_id,
          slug,
        },
      });

      if (existingPermission) {
        await prisma.permissions.update({
          where: { id: existingPermission.id },
          data: { slug, company_id, entity_id: masterEntityId },
        });
        console.log(`🔄 Updated Permission: ${slug}`);
      } else {
        await prisma.permissions.create({
          data: {
            slug,
            company_id,
            name: right,
            entity_id: masterEntityId,
            order_no,
          },
        });
        console.log(`✅ Created Permission: ${slug}`);
      }

      order_no++;
    }

    console.log('✅ Permission creation completed.');
  } catch (error) {
    console.log(`❌ Permission seeder failed: ${permissionData.name}`);
    throw error;
  }
}

export async function permissionsSeeder(company_id: number) {
  try {
    console.log("🚀 Starting Permissions Seeder...");

    const permissions = JSON.parse(
      fs.readFileSync(path.join(__dirname, 'json/permission-seeds.json'), 'utf-8')
    );

    for (const permissionData of permissions) {
      await findOrCreatePermission(permissionData, company_id);
    }

    console.log('✅ Permissions creation completed.');
  } catch (error) {
    console.error(`❌ Permissions Seeder Failed: ${error}`);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
