import { PrismaClient, Prisma } from '@prisma/sql/client';
import { v4 as uuidv4 } from 'uuid';

const prisma = new PrismaClient();

export async function languageSeeds(companyId: number) {
  try {
    console.log("🚀 Starting Language Seeder...");

    let user = await prisma.user.findFirst({
      where: { company_id: companyId, role: 'super_admin' },
    });

    if (!user) {
      console.log(`❌ Language seeder failed: No super_admin user for company_id ${companyId}`);
      return false;
    }

    const createdBy = user.id;
    const langList = [
      { name: "English", code: "en-GB", status: 1 },
      { name: "Arabic", code: "ar-QA", status: 1 },
      { name: "French", code: "fr-FR", status: 3 },
      { name: "German", code: "de-DE", status: 3 },
    ];

    const company = await prisma.user.findFirst({
      where: { company_id: companyId },
    });

    if (!company) {
      console.log(`❌ WARNING: Company with id ${companyId} does not exist.`);
      return;
    }

    for (const { name, code, status } of langList) {
      const existingRecord = await prisma.language.findFirst({
        where: {
          name: name,
          company_id: companyId,
        },
      });

      if (existingRecord) {
        console.log(`🔄 Language "${name}" already exists for company_id ${companyId}`);
      } else {
        await prisma.language.create({
          data: {
            uuid: uuidv4(), // Use UUID generation for SQL Server
            name,
            name_in_english: name,
            code,
            description: name,
            status_id: status,
            company_id: companyId,
            created_by: createdBy,
          },
        });
        console.log(`✅ Created new language: name ${name}, company_id ${companyId}`);
      }
    }

    console.log("✅ Language Seeder completed successfully.");
  } catch (error) {
    console.error(`❌ Error in languageSeeds: ${error}`);
    if (error instanceof Prisma.PrismaClientKnownRequestError && error.code === 'P2003' && error.meta) {
      console.error(`Foreign key constraint failed: ${error.meta.field_name}`);
    }
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
