import { PrismaClient} from '@prisma/sql/client';
const prisma = new PrismaClient();

async function findOrCreateProcess(process: any, company_id: number) {
  try {
    let templateProcess = await prisma.emailTemplateProcess.findFirst({
      where: { company_id, slug: process.slug },
    });

    let data = {
      ...process,
      company_id,
      query_information: process.query_information ? JSON.stringify(process.query_information) : "",
    };

    if (!templateProcess) {
      templateProcess = await prisma.emailTemplateProcess.create({ data });
      console.log(`✅ Email Template Process created: ${templateProcess.name}`);
    } else {
      await prisma.emailTemplateProcess.update({
        where: { id: templateProcess.id },
        data,
      });
      console.log(`🔄 Email Template Process updated: ${templateProcess.name}`);
    }
  } catch (error) {
    console.error(`❌ Email Template Process seeder failed: ${process.name}`, error);
    throw error;
  }
}

async function findOrCreateEmailTags(tag: any, company_id: number) {
  try {
    let templateTag = await prisma.emailTemplateTag.findFirst({
      where: { company_id, slug: tag.slug },
    });

    if (!templateTag) {
      templateTag = await prisma.emailTemplateTag.create({ data: { ...tag, company_id } });
      console.log(`✅ Email Template Tag created: ${tag.slug}`);
    } else {
      await prisma.emailTemplateTag.update({ where: { id: templateTag.id }, data: tag });
      console.log(`🔄 Email Template Tag updated: ${tag.slug}`);
    }
  } catch (error) {
    console.error(`❌ Email Template Tag seeder failed: ${tag.slug}`, error);
    throw error;
  }
}

async function findOrCreateEmailRecipientTags(tag: any, company_id: number) {
  try {
    let templateTag = await prisma.emailTemplateRecipientTag.findFirst({
      where: { company_id, slug: tag.slug },
    });

    let data = {
      ...tag,
      company_id,
      query_information: tag.query_information ? JSON.stringify(tag.query_information) : "",
    };

    if (!templateTag) {
      templateTag = await prisma.emailTemplateRecipientTag.create({ data });
      console.log(`✅ Email Template Recipient Tag created: ${tag.slug}`);
    } else {
      await prisma.emailTemplateRecipientTag.update({
        where: { id: templateTag.id },
        data,
      });
      console.log(`🔄 Email Template Recipient Tag updated: ${templateTag.slug}`);
    }
  } catch (error) {
    console.error(`❌ Email Template Recipient Tag seeder failed: ${tag.slug}`, error);
    throw error;
  }
}

async function findOrCreateEmailTagsAndTemplateMap(map: any, company_id: number) {
  try {
    const emailTemplateProcess = await prisma.emailTemplateProcess.findFirst({
      where: { company_id, slug: map.email_template_process },
    });
    const emailTemplateTag = await prisma.emailTemplateTag.findFirst({
      where: { company_id, slug: map.email_template_tag },
    });

    if (emailTemplateProcess && emailTemplateTag) {
      let templateGroup = await prisma.emailTemplateProcessTagsMapping.findFirst({
        where: {
          company_id,
          email_template_process_id: emailTemplateProcess.id,
          email_template_tag_id: emailTemplateTag.id,
        },
      });

      if (!templateGroup) {
        await prisma.emailTemplateProcessTagsMapping.create({
          data: {
            company_id,
            email_template_process_id: emailTemplateProcess.id,
            email_template_tag_id: emailTemplateTag.id,
          },
        });
        console.log(`✅ Email Template group map created: ${emailTemplateProcess.id}, ${emailTemplateTag.id}, ${company_id}`);
      } else {
        console.log(`🔄 Email Template group map already exists: ${emailTemplateProcess.id}, ${emailTemplateTag.id}, ${company_id}`);
      }
    }
  } catch (error) {
    console.error(`❌ Email Template group map seeder failed: ${map.email_template_process}, ${map.email_template_tag}`, error);
    throw error;
  }
}

export async function emailTemplatesProcessSeeder(company_id: number) {
  try {
    console.log("🚀 Starting Email Templates Process Seeder...");

    const company = await prisma.user.findFirst({
      where: { company_id },
    });

    if (!company) {
      console.log(`❌ WARNING: Company with id ${company_id} does not exist.`);
      return;
    }

    const processes = [
      {
        "name": "Forgot Password",
        "slug": "forgot-password",
        "status_id": 1,
        "query_information": {
          "company_id": 1,
          "search_all": [
              {
                  "value": "@process.unique_id",
                  "operator": "=",
                  "column_name": "otps.id"
              }
          ],
          "limit_range": 1,
          "print_query": false,
          "start_index": 0,
          "sort_columns": [
              [
                  "otps.id",
                  "desc"
              ]
          ],
          "primary_table": "users",
          "select_columns": [
              ["username"], ["user_details.user_id"], ["last_name", "user_lastname"], ["email", "user_email"], ["first_name", "user_firstname"], ["CONCAT(first_name, ' ', last_name)", "user_fullname"], ["phone_number", "user_phone"], ["otp"], ["STRING_AGG(roles.name, ', ')", "user_roles"]
          ],
        "includes": [
          {
            "table_name": "user_details",
            "join_type": "INNER",
            "join_condition": "users.id = user_details.user_id"
          },
          {
            "table_name": "otps",
            "join_type": "INNER",
            "join_condition": "users.id = otps.user_id"
          },
          {
            "table_name": "user_roles",
            "join_type": "LEFT",
            "join_condition": "users.id = user_roles.user_id"
          },
          {
            "table_name": "roles",
            "join_type": "LEFT",
            "join_condition": "roles.id = user_roles.role_id"
          }
        ],
        "group_by":["username", "user_details.user_id", "last_name", "email", "first_name", "phone_number", "otp", "otps.id"]
      }
      },
      {
        "name": "Email Verify",
        "slug": "email-verify",
        "status_id": 1,
        "query_information": {
          "company_id": 1,
          "search_all": [
              {
                  "value": "@process.unique_id",
                  "operator": "=",
                  "column_name": "otps.id"
              }
          ],
          "limit_range": 1,
          "print_query": false,
          "start_index": 0,
          "sort_columns": [
              [
                  "otps.id",
                  "desc"
              ]
          ],
          "primary_table": "users",
          "select_columns": [
              ["username"], ["user_details.user_id"], ["last_name", "user_lastname"], ["email", "user_email"], ["first_name", "user_firstname"], ["CONCAT(first_name, ' ', last_name)", "user_fullname"], ["phone_number", "user_phone"], ["otp"], ["STRING_AGG(roles.name, ', ')", "user_roles"]
          ],
        "includes": [
          {
            "table_name": "user_details",
            "join_type": "INNER",
            "join_condition": "users.id = user_details.user_id"
          },
          {
            "table_name": "otps",
            "join_type": "INNER",
            "join_condition": "users.id = otps.user_id"
          },
          {
            "table_name": "user_roles",
            "join_type": "LEFT",
            "join_condition": "users.id = user_roles.user_id"
          },
          {
            "table_name": "roles",
            "join_type": "LEFT",
            "join_condition": "roles.id = user_roles.role_id"
          }
        ],
        "group_by":["username", "user_details.user_id", "last_name", "email", "first_name", "phone_number", "otp", "otps.id"]
      }
      },
      {
        "name": "Login OTP",
        "slug": "login-otp",
        "status_id": 1,
        "query_information": {
          "company_id": 1,
          "search_all": [
              {
                  "value": "@process.unique_id",
                  "operator": "=",
                  "column_name": "otps.id"
              }
          ],
          "limit_range": 1,
          "print_query": false,
          "start_index": 0,
          "sort_columns": [
              [
                  "otps.id",
                  "desc"
              ]
          ],
          "primary_table": "users",
          "select_columns": [
              ["username"], ["user_details.user_id"], ["last_name", "user_lastname"], ["email", "user_email"], ["first_name", "user_firstname"], ["CONCAT(first_name, ' ', last_name)", "user_fullname"], ["phone_number", "user_phone"], ["otp"], ["STRING_AGG(roles.name, ', ')", "user_roles"]
          ],
        "includes": [
          {
            "table_name": "user_details",
            "join_type": "INNER",
            "join_condition": "users.id = user_details.user_id"
          },
          {
            "table_name": "otps",
            "join_type": "INNER",
            "join_condition": "users.id = otps.user_id"
          },
          {
            "table_name": "user_roles",
            "join_type": "LEFT",
            "join_condition": "users.id = user_roles.user_id"
          },
          {
            "table_name": "roles",
            "join_type": "LEFT",
            "join_condition": "roles.id = user_roles.role_id"
          }
        ],
        "group_by":["username", "user_details.user_id", "last_name", "email", "first_name", "phone_number", "otp", "otps.id"]
      }
      },
      {
        "name": "Password Changed",
        "slug": "password-changed",
        "status_id": 1,
        "query_information": {
          "company_id": 1,
          "search_all": [
              {
                  "value": "@process.unique_id",
                  "operator": "=",
                  "column_name": "users.id"
              }
          ],
          "limit_range": 1,
          "print_query": false,
          "start_index": 0,
          "sort_columns": [
              [
                  "users.email",
                  "asc"
              ]
          ],
          "primary_table": "users",
          "select_columns": [
              ["username"], ["user_details.user_id"], ["last_name", "user_lastname"], ["email", "user_email"], ["first_name", "user_firstname"], ["CONCAT(first_name, ' ', last_name)", "user_fullname"], ["phone_number", "user_phone"], ["STRING_AGG(roles.name, ', ')", "user_roles"]
          ],
        "includes": [
          {
            "table_name": "user_details",
            "join_type": "INNER",
            "join_condition": "users.id = user_details.user_id"
          },
          {
            "table_name": "user_roles",
            "join_type": "LEFT",
            "join_condition": "users.id = user_roles.user_id"
          },
          {
            "table_name": "roles",
            "join_type": "LEFT",
            "join_condition": "roles.id = user_roles.role_id"
          }
        ],
        "group_by":["username", "user_details.user_id", "last_name", "email", "first_name", "phone_number", "users.id"]
      }
      },
      {
        "name": "Profile Updated",
        "slug": "profile-updated",
        "status_id": 1,
        "query_information": {
          "company_id": 1,
          "search_all": [
              {
                  "value": "@process.unique_id",
                  "operator": "=",
                  "column_name": "users.id"
              }
          ],
          "limit_range": 1,
          "print_query": false,
          "start_index": 0,
          "sort_columns": [
              [
                  "users.email",
                  "asc"
              ]
          ],
          "primary_table": "users",
          "select_columns": [
              ["username"], ["user_details.user_id"], ["last_name", "user_lastname"], ["email", "user_email"], ["first_name", "user_firstname"], ["CONCAT(first_name, ' ', last_name)", "user_fullname"], ["phone_number", "user_phone"], ["STRING_AGG(roles.name, ', ')", "user_roles"]
          ],
        "includes": [
          {
            "table_name": "user_details",
            "join_type": "INNER",
            "join_condition": "users.id = user_details.user_id"
          },
          {
            "table_name": "user_roles",
            "join_type": "LEFT",
            "join_condition": "users.id = user_roles.user_id"
          },
          {
            "table_name": "roles",
            "join_type": "LEFT",
            "join_condition": "roles.id = user_roles.role_id"
          }
        ],
        "group_by":["username", "user_details.user_id", "last_name", "email", "first_name", "phone_number", "users.id"]
      }
      },
      {
        "name": "User Created",
        "slug": "user-created",
        "status_id": 1,
        "query_information": {
          "company_id": 1,
          "search_all": [
              {
                  "value": "@process.unique_id",
                  "operator": "=",
                  "column_name": "users.id"
              }
          ],
          "limit_range": 1,
          "print_query": false,
          "start_index": 0,
          "sort_columns": [
              [
                  "users.email",
                  "asc"
              ]
          ],
          "primary_table": "users",
          "select_columns": [
              ["username"], ["user_details.user_id"], ["last_name", "user_lastname"], ["email", "user_email"], ["first_name", "user_firstname"], ["CONCAT(first_name, ' ', last_name)", "user_fullname"], ["phone_number", "user_phone"], ["STRING_AGG(roles.name, ', ')", "user_roles"]
          ],
        "includes": [
          {
            "table_name": "user_details",
            "join_type": "INNER",
            "join_condition": "users.id = user_details.user_id"
          },
          {
            "table_name": "user_roles",
            "join_type": "LEFT",
            "join_condition": "users.id = user_roles.user_id"
          },
          {
            "table_name": "roles",
            "join_type": "LEFT",
            "join_condition": "roles.id = user_roles.role_id"
          }
        ],
        "group_by":["username", "user_details.user_id", "last_name", "email", "first_name", "phone_number", "users.id"]
      }
      }
    ];

    await Promise.all(processes.map(process => findOrCreateProcess(process, company_id)));

    const tags = [
      { name: "User First Name", slug: "[user_firstname]", status_id: 1, description: "Returns first name of user" },
      { name: "User Full Name", slug: "[user_fullname]", status_id: 1, description: "Returns full name of user" },
      { name: "OTP", slug: "[otp]", status_id: 1, description: "OTP value" },
      { name: "User Email", slug: "[user_email]", status_id: 1, description: "User's email address" },
    ];

    await Promise.all(tags.map(tag => findOrCreateEmailTags(tag, company_id)));

    const emailProcessAndTagGroup = [
      { email_template_process: "forgot-password", email_template_tag: "[otp]" },
      { email_template_process: "email-verify", email_template_tag: "[otp]" },
      { email_template_process: "login-otp", email_template_tag: "[otp]" },
      { email_template_process: "forgot-password", email_template_tag: "[user_fullname]" },
    ];

    await Promise.all(emailProcessAndTagGroup.map(map => findOrCreateEmailTagsAndTemplateMap(map, company_id)));

    const recipientTags = [
      {
        "description": "User who requested the api",
        "slug": "requested_user",
        "name": "requested_user",
        "status_id": 1,
        "query_information": {
          "company_id": 1,
          "search_all": [
              {
                  "value": "3",
                  "operator": "!=",
                  "column_name": "users.status_id"
              },
              {
                  "value": "@process.user_id",
                  "operator": "=",
                  "column_name": "users.id"
              }
          ],
          "limit_range": 25,
          "print_query": false,
          "start_index": 0,
          "sort_columns": [
              [
                  "users.email",
                  "asc"
              ]
          ],
          "primary_table": "users",
          "select_columns": [
              ["user_id"], ["email"], ["email", "email_to"], ["first_name"], ["CONCAT(first_name, ' ', last_name)", "full_name"], ["role"]
          ],
        "includes": [
          {
            "table_name": "user_details",
            "join_type": "INNER",
            "join_condition": "users.id = user_details.user_id"
          }
        ]
      }
      },
      {
        "description": "Super admin of the user who requested the api",
        "slug": "super_admin",
        "name": "super_admin",
        "status_id": 1,
        "query_information": {
          "primary_table": "users",
          "select_columns": [
            ["b.email", "email"]
          ],
          "includes": [
            {
              "table_name": "users as b",
              "join_type": "LEFT",
              "join_condition": "users.company_id = b.company_id AND b.role='super_admin'"
            }
          ],
          "search_all": [
            {
              "column_name": "b.status_id",
              "operator": "=",
              "value": 1
            },
            {
              "column_name": "users.id",
              "operator": "=",
              "value": "@process.user_id"
            }
          ],
          "company_id": 1,
          "start_index": 0,
          "print_query": false
        }
        
      },
      {
        "description": "Department Head of the user who requested the api",
        "slug": "department_head",
        "name": "department_head",
        "status_id": 1,
        "query_information": {
          "primary_table": "user_details",
          "select_columns": [
            ["users.email", "email"]
          ],
          "includes": [
              {
                  "table_name": "departments as d",
                  "join_type": "INNER",
                  "join_condition": "user_details.department_id = d.id"
              },
              {
                  "table_name": "users",
                  "join_type": "INNER",
                  "join_condition": "users.id = d.department_head_id"
              }
          ],
          "search_all": [
              {
                  "column_name": "users.status_id",
                  "operator": "=",
                  "value": "1"
              },
              {
                  "column_name": "user_details.id",
                  "operator": "=",
                  "value": "@process.user_id"
              }
          ],
          "company_id": 1,
          "limit_range": 10,
          "start_index": 0,
          "print_query": false
      }        
      }
    ];

    await Promise.all(recipientTags.map(tag => findOrCreateEmailRecipientTags(tag, company_id)));

    console.log("✅ Email Templates Process Seeder completed successfully.");
  } catch (error) {
    console.error(`❌ Error in emailTemplatesProcessSeeder: ${error}`);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
