import { PrismaClient } from '@prisma/pg/client';
import fs from 'fs';
const prisma = new PrismaClient();

export async function WarehouseSeeder(company_id: number) {
  try {
    const rows = JSON.parse(
      fs.readFileSync(__dirname + '/json/warehouse-seeds.json', 'utf-8'),
    );

    const user = await prisma.user.findFirst({
      where: { company_id: company_id, role: 'super_admin' },
    });

    if (!user?.id) {
      throw new Error('Super admin user not found!');
    }

    for (const row of rows) {
      // Check if the Warehouse already exists by its code and company_id
      const existingWarehouse = await prisma.warehouses.findUnique({
        where: { company_id_code: { company_id, code: row.code } },
      });

      if (existingWarehouse) {
        // Update the Warehouse if it already exists
        await prisma.warehouses.update({
          where: { id: existingWarehouse.id },
          data: {
            name: row.name,
            region_id: row.region_id,
            status_id: row.status_id || 1, // Default to active status if not provided
            updated_by: user.id, // Ensure updated_by is assigned
          },
        });
        console.log(`Updated Warehouse with code ${row.code}`);
      } else {
        // Create a new Warehouse if it doesn't exist
        await prisma.warehouses.create({
          data: {
            code: row.code,
            name: row.name,
            region_id: row.region_id,
            company_id,
            status_id: row.status_id || 1, // Default to active status if not provided
            created_by: user.id, // Ensure created_by is assigned
            updated_by: user.id, // Ensure updated_by is assigned
          },
        });
        console.log(`Created Warehouse with code ${row.code}`);
      }
    }
  } catch (error) {
    console.error(`Error in WarehouseSeeder: ${error}`);
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}
