import { PrismaClient, $Enums } from '@prisma/pg/client';

const prisma = new PrismaClient();

interface EmailTemplateData {
  email_template_process_slug: string;
  template_slug?: string;
  recipient_type?: $Enums.RecipientType;
  email_to?: string;
}

export async function appEmailTemplateAssignmentsSeeder(company_id: number) {
  try {
    let user = await prisma.user.findFirst({
      where: { company_id: company_id, role: 'super_admin' },
    });
    if (!user) {
      console.log(
        `Menu seeder failed: admin user not available for company id ${company_id}`,
      );
      return false;
    }
    const created_by = user.id;

    const datas: EmailTemplateData[] = [
      {
        email_template_process_slug: 'mtr-mail-import',
        template_slug: 'mtr-mail-import',
        recipient_type: 'tag',
        email_to: 'requested_user',
      },
      {
        email_template_process_slug: 'mir-mail-import',
        template_slug: 'mir-mail-import',
        recipient_type: 'tag',
        email_to: 'requested_user',
      },
    ];

    for (const data of datas) {
      // Fetch the email_template_process_id using the provided slug and company_id
      const emailTemplateProcess = await prisma.emailTemplateProcess.findFirst({
        where: {
          slug: data.email_template_process_slug,
          company_id: company_id,
        },
      });

      if (!emailTemplateProcess) {
        console.error(
          `Email template process with slug "${data.email_template_process_slug}" not found for company_id ${company_id}.`,
        );
        continue;
      }

      // Initialize template IDs
      let template_id: number | null = null;

      // Fetch the user_template_id if template_slug is provided
      if (data.template_slug) {
        const template = await prisma.emailTemplate.findFirst({
          where: {
            slug: data.template_slug,
            company_id: company_id,
          },
        });
        if (template) {
          template_id = template.id;
        } else {
          console.error(
            `User template with slug "${data.template_slug}" not found for company_id ${company_id}.`,
          );
          continue;
        }
      }

      let email_to: string | null = `${data.email_to}` || null;
      if (data.recipient_type === 'tag') {
        email_to = null;
        if (data.email_to) {
          const template = await prisma.emailTemplateRecipientTag.findFirst({
            where: {
              slug: `${data.email_to}`,
              company_id: company_id,
            },
          });
          if (template) {
            email_to = `${template.id}`;
          } else {
            console.error(
              `Email template recipient tag with slug "${data.email_to}" not found for company_id ${company_id}.`,
            );
            continue;
          }
        }
      }
      const email_template_process_id = emailTemplateProcess.id;

      const existingAssignment = await prisma.emailTemplateAssignment.findFirst(
        {
          where: {
            email_template_process_id: email_template_process_id,
            template_id: template_id || 0,
            recipient_type: data.recipient_type,
            email_to,
            company_id: company_id,
          },
        },
      );

      if (!existingAssignment && template_id) {
        await prisma.emailTemplateAssignment.create({
          data: {
            email_template_process_id,
            template_id,
            recipient_type: data.recipient_type,
            email_to,
            company_id,
            created_by,
            created_at: new Date(),
          },
        });
        console.log(
          `Email Template Assignment with email_template_process_id "${email_template_process_id}" created for company_id ${company_id}.`,
        );
      } else {
        console.log(
          `Email Template Assignment with email_template_process_id "${email_template_process_id}" already exists for company_id ${company_id}.`,
        );
      }
    }

    console.log('Email Template Assignment creation completed.');
  } catch (error) {
    console.error('Error creating email template assignments:', error);
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}
