import {
  EmailTemplateProcessTagsMapping,
  EmailTemplateProcess,
  EmailTemplateTag,
  PrismaClient,
  EmailTemplateRecipientTag,
} from '@prisma/pg/client';
const prisma = new PrismaClient();

async function findOrCreateProcess(process: any, company_id: number) {
  try {
    let templateProcess: EmailTemplateProcess | null =
      await prisma.emailTemplateProcess.findUnique({
        where: { company_id_slug: { company_id, slug: process.slug } },
      });

    if (!templateProcess) {
      let data = {
        ...process,
        company_id,
        query_information: '',
      };
      if (process.query_information) {
        data.query_information = JSON.stringify(process.query_information);
      }

      templateProcess = await prisma.emailTemplateProcess.create({
        data,
      });
      console.log(
        `Email Template Process created successfully: ${templateProcess.name}`,
      );
    } else {
      let data = {
        ...process,
        query_information: '',
      };
      if (process.query_information) {
        data.query_information = JSON.stringify(process.query_information);
      }

      templateProcess = await prisma.emailTemplateProcess.update({
        where: { company_id_slug: { company_id, slug: process.slug } },
        data,
      });
      console.log(
        `Email Template Process updated successfully: ${templateProcess.name}`,
      );
    }

    return templateProcess;
  } catch (error) {
    console.log(`Email Template Process seeder failed: ${process.name}`);
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}

async function findOrCreateEmailTags(tag: any, company_id: number) {
  try {
    let templateTag: EmailTemplateTag | null =
      await prisma.emailTemplateTag.findUnique({
        where: { company_id_slug: { company_id, slug: tag.slug } },
      });

    if (!templateTag) {
      let data = {
        ...tag,
        company_id,
      };

      templateTag = await prisma.emailTemplateTag.create({
        data,
      });
      console.log(`Email Template Tag created successfully: ${tag.slug}`);
    } else {
      let data = {
        ...tag,
      };

      templateTag = await prisma.emailTemplateTag.update({
        where: { company_id_slug: { company_id, slug: tag.slug } },
        data,
      });
      console.log(
        `Email Template Tag updated successfully: ${templateTag.slug}`,
      );
    }

    return templateTag;
  } catch (error) {
    console.log(`Email Template Tag seeder failed: ${tag.slug}`);
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}

async function findOrCreateEmailRecipientTags(tag: any, company_id: number) {
  try {
    let templateTag: EmailTemplateRecipientTag | null =
      await prisma.emailTemplateRecipientTag.findUnique({
        where: { company_id_slug: { company_id, slug: tag.slug } },
      });

    if (!templateTag) {
      let data = {
        ...tag,
        company_id,
        query_information: '',
      };
      if (tag.query_information) {
        data.query_information = JSON.stringify(tag.query_information);
      }

      templateTag = await prisma.emailTemplateRecipientTag.create({
        data,
      });
      console.log(
        `Email Template Recipient Tag created successfully: ${tag.slug}`,
      );
    } else {
      let data = {
        ...tag,
        query_information: '',
      };
      if (tag.query_information) {
        data.query_information = JSON.stringify(tag.query_information);
      }

      templateTag = await prisma.emailTemplateRecipientTag.update({
        where: { company_id_slug: { company_id, slug: tag.slug } },
        data,
      });
      console.log(
        `Email Template Recipient Tag updated successfully: ${templateTag.slug}`,
      );
    }

    return templateTag;
  } catch (error) {
    console.log(`Email Template Recipient Tag seeder failed: ${tag.slug}`);
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}

async function findOrCreateEmailTagsAndTemplateMap(
  map: any,
  company_id: number,
) {
  try {
    const email_template_process = await prisma.emailTemplateProcess.findUnique(
      {
        where: {
          company_id_slug: { company_id, slug: map.email_template_process },
        },
      },
    );
    const email_template_tag = await prisma.emailTemplateTag.findUnique({
      where: { company_id_slug: { company_id, slug: map.email_template_tag } },
    });
    if (email_template_process && email_template_tag) {
      let templateGroup: EmailTemplateProcessTagsMapping | null =
        await prisma.emailTemplateProcessTagsMapping.findUnique({
          where: {
            email_template_process_id_email_template_tag_id_company_id: {
              email_template_process_id: email_template_process.id,
              email_template_tag_id: email_template_tag.id,
              company_id,
            },
          },
        });

      if (!templateGroup) {
        let data = {
          email_template_process_id: email_template_process.id,
          email_template_tag_id: email_template_tag.id,
          company_id,
        };

        templateGroup = await prisma.emailTemplateProcessTagsMapping.create({
          data,
        });
        console.log(
          `Email Template group map created successfully: ${email_template_process.id}, ${email_template_tag.id}, ${company_id}`,
        );
      } else {
        console.log(
          `Email Template group map already exist: ${email_template_process.id}, ${email_template_tag.id}, ${company_id}`,
        );
      }

      return templateGroup;
    }
    return false;
  } catch (error) {
    console.log(
      `Email Template group map seeder failed: ${map.email_template_process}, ${map.email_template_tag}, ${company_id}`,
    );
    throw error;
  } finally {
    // await prisma.$disconnect();
  }
}

export async function emailTemplatesProcessSeeder(company_id: number) {
  try {
    const company = await prisma.user.findFirst({
      where: { company_id: company_id },
    });

    if (!company) {
      console.log(
        `WARNING: emailTemplatesProcessSeeder: Company with id ${company_id} does not exist.`,
      );
      return;
    }

    const datas = [
      {
        name: 'Forgot Password',
        slug: 'forgot-password',
        status_id: 1,
        query_information: {
          company_id: 1,
          search_all: [
            {
              value: '@process.unique_id',
              operator: '=',
              column_name: 'otps.id',
            },
          ],
          limit_range: 1,
          print_query: false,
          start_index: 0,
          sort_columns: [['otps.id', 'desc']],
          primary_table: 'users',
          select_columns: [
            ['username'],
            ['user_details.user_id'],
            ['last_name', 'user_lastname'],
            ['email', 'user_email'],
            ['first_name', 'user_firstname'],
            ["CONCAT(first_name, ' ', last_name)", 'user_fullname'],
            ['phone_number', 'user_phone'],
            ['otp'],
            ["STRING_AGG(roles.name, ', ')", 'user_roles'],
          ],
          includes: [
            {
              table_name: 'user_details',
              join_type: 'INNER',
              join_condition: 'users.id = user_details.user_id',
            },
            {
              table_name: 'otps',
              join_type: 'INNER',
              join_condition: 'users.id = otps.user_id',
            },
            {
              table_name: 'user_roles',
              join_type: 'LEFT',
              join_condition: 'users.id = user_roles.user_id',
            },
            {
              table_name: 'roles',
              join_type: 'LEFT',
              join_condition: 'roles.id = user_roles.role_id',
            },
          ],
          group_by: [
            'username',
            'user_details.user_id',
            'last_name',
            'email',
            'first_name',
            'phone_number',
            'otp',
            'otps.id',
          ],
        },
      },
      {
        name: 'Email Verify',
        slug: 'email-verify',
        status_id: 1,
        query_information: {
          company_id: 1,
          search_all: [
            {
              value: '@process.unique_id',
              operator: '=',
              column_name: 'otps.id',
            },
          ],
          limit_range: 1,
          print_query: false,
          start_index: 0,
          sort_columns: [['otps.id', 'desc']],
          primary_table: 'users',
          select_columns: [
            ['username'],
            ['user_details.user_id'],
            ['last_name', 'user_lastname'],
            ['email', 'user_email'],
            ['first_name', 'user_firstname'],
            ["CONCAT(first_name, ' ', last_name)", 'user_fullname'],
            ['phone_number', 'user_phone'],
            ['otp'],
            ["STRING_AGG(roles.name, ', ')", 'user_roles'],
          ],
          includes: [
            {
              table_name: 'user_details',
              join_type: 'INNER',
              join_condition: 'users.id = user_details.user_id',
            },
            {
              table_name: 'otps',
              join_type: 'INNER',
              join_condition: 'users.id = otps.user_id',
            },
            {
              table_name: 'user_roles',
              join_type: 'LEFT',
              join_condition: 'users.id = user_roles.user_id',
            },
            {
              table_name: 'roles',
              join_type: 'LEFT',
              join_condition: 'roles.id = user_roles.role_id',
            },
          ],
          group_by: [
            'username',
            'user_details.user_id',
            'last_name',
            'email',
            'first_name',
            'phone_number',
            'otp',
            'otps.id',
          ],
        },
      },
      {
        name: 'Login OTP',
        slug: 'login-otp',
        status_id: 1,
        query_information: {
          company_id: 1,
          search_all: [
            {
              value: '@process.unique_id',
              operator: '=',
              column_name: 'otps.id',
            },
          ],
          limit_range: 1,
          print_query: false,
          start_index: 0,
          sort_columns: [['otps.id', 'desc']],
          primary_table: 'users',
          select_columns: [
            ['username'],
            ['user_details.user_id'],
            ['last_name', 'user_lastname'],
            ['email', 'user_email'],
            ['first_name', 'user_firstname'],
            ["CONCAT(first_name, ' ', last_name)", 'user_fullname'],
            ['phone_number', 'user_phone'],
            ['otp'],
            ["STRING_AGG(roles.name, ', ')", 'user_roles'],
          ],
          includes: [
            {
              table_name: 'user_details',
              join_type: 'INNER',
              join_condition: 'users.id = user_details.user_id',
            },
            {
              table_name: 'otps',
              join_type: 'INNER',
              join_condition: 'users.id = otps.user_id',
            },
            {
              table_name: 'user_roles',
              join_type: 'LEFT',
              join_condition: 'users.id = user_roles.user_id',
            },
            {
              table_name: 'roles',
              join_type: 'LEFT',
              join_condition: 'roles.id = user_roles.role_id',
            },
          ],
          group_by: [
            'username',
            'user_details.user_id',
            'last_name',
            'email',
            'first_name',
            'phone_number',
            'otp',
            'otps.id',
          ],
        },
      },
      {
        name: 'Password Changed',
        slug: 'password-changed',
        status_id: 1,
        query_information: {
          company_id: 1,
          search_all: [
            {
              value: '@process.unique_id',
              operator: '=',
              column_name: 'users.id',
            },
          ],
          limit_range: 1,
          print_query: false,
          start_index: 0,
          sort_columns: [['users.email', 'asc']],
          primary_table: 'users',
          select_columns: [
            ['username'],
            ['user_details.user_id'],
            ['last_name', 'user_lastname'],
            ['email', 'user_email'],
            ['first_name', 'user_firstname'],
            ["CONCAT(first_name, ' ', last_name)", 'user_fullname'],
            ['phone_number', 'user_phone'],
            ["STRING_AGG(roles.name, ', ')", 'user_roles'],
          ],
          includes: [
            {
              table_name: 'user_details',
              join_type: 'INNER',
              join_condition: 'users.id = user_details.user_id',
            },
            {
              table_name: 'user_roles',
              join_type: 'LEFT',
              join_condition: 'users.id = user_roles.user_id',
            },
            {
              table_name: 'roles',
              join_type: 'LEFT',
              join_condition: 'roles.id = user_roles.role_id',
            },
          ],
          group_by: [
            'username',
            'user_details.user_id',
            'last_name',
            'email',
            'first_name',
            'phone_number',
            'users.id',
          ],
        },
      },
      {
        name: 'Profile Updated',
        slug: 'profile-updated',
        status_id: 1,
        query_information: {
          company_id: 1,
          search_all: [
            {
              value: '@process.unique_id',
              operator: '=',
              column_name: 'users.id',
            },
          ],
          limit_range: 1,
          print_query: false,
          start_index: 0,
          sort_columns: [['users.email', 'asc']],
          primary_table: 'users',
          select_columns: [
            ['username'],
            ['user_details.user_id'],
            ['last_name', 'user_lastname'],
            ['email', 'user_email'],
            ['first_name', 'user_firstname'],
            ["CONCAT(first_name, ' ', last_name)", 'user_fullname'],
            ['phone_number', 'user_phone'],
            ["STRING_AGG(roles.name, ', ')", 'user_roles'],
          ],
          includes: [
            {
              table_name: 'user_details',
              join_type: 'INNER',
              join_condition: 'users.id = user_details.user_id',
            },
            {
              table_name: 'user_roles',
              join_type: 'LEFT',
              join_condition: 'users.id = user_roles.user_id',
            },
            {
              table_name: 'roles',
              join_type: 'LEFT',
              join_condition: 'roles.id = user_roles.role_id',
            },
          ],
          group_by: [
            'username',
            'user_details.user_id',
            'last_name',
            'email',
            'first_name',
            'phone_number',
            'users.id',
          ],
        },
      },
      {
        name: 'User Created',
        slug: 'user-created',
        status_id: 1,
        query_information: {
          company_id: 1,
          search_all: [
            {
              value: '@process.unique_id',
              operator: '=',
              column_name: 'users.id',
            },
          ],
          limit_range: 1,
          print_query: false,
          start_index: 0,
          sort_columns: [['users.email', 'asc']],
          primary_table: 'users',
          select_columns: [
            ['username'],
            ['user_details.user_id'],
            ['last_name', 'user_lastname'],
            ['email', 'user_email'],
            ['first_name', 'user_firstname'],
            ["CONCAT(first_name, ' ', last_name)", 'user_fullname'],
            ['phone_number', 'user_phone'],
            ["STRING_AGG(roles.name, ', ')", 'user_roles'],
          ],
          includes: [
            {
              table_name: 'user_details',
              join_type: 'INNER',
              join_condition: 'users.id = user_details.user_id',
            },
            {
              table_name: 'user_roles',
              join_type: 'LEFT',
              join_condition: 'users.id = user_roles.user_id',
            },
            {
              table_name: 'roles',
              join_type: 'LEFT',
              join_condition: 'roles.id = user_roles.role_id',
            },
          ],
          group_by: [
            'username',
            'user_details.user_id',
            'last_name',
            'email',
            'first_name',
            'phone_number',
            'users.id',
          ],
        },
      },
      {
        name: 'Mail Import',
        slug: 'mail-import',
        status_id: 1,
        query_information: {
          company_id: 1,
          search_all: [
            {
              value: '@process.user_id',
              operator: '=',
              column_name: 'users.id',
            },
          ],
          limit_range: 1,
          print_query: false,
          start_index: 0,
          sort_columns: [['users.email', 'asc']],
          primary_table: 'users',
          select_columns: [
            ['user_details.user_id'],
            ['email', 'user_email'],
            ["CONCAT(first_name, ' ', last_name)", 'user_fullname'],
          ],
          includes: [
            {
              table_name: 'user_details',
              join_type: 'INNER',
              join_condition: 'users.id = user_details.user_id',
            },
          ],
          group_by: [
            'user_details.user_id',
            'last_name',
            'email',
            'first_name',
            'users.id',
          ],
        },
      },
      {
        name: 'Approval User Mail',
        slug: 'approval-user-mail',
        status_id: 1,
        query_information: {
          company_id:1,
          search_all:[
             {
                "value":"@process.unique_id",
                "operator":"=",
                "column_name":"users.id"
             }
          ],
          limit_range:1,
          print_query:false,
          start_index:0,
          sort_columns:[
             [
                "users.email",
                "asc"
             ]
          ],
          primary_table:"users",
          select_columns:[
             [
                "email",
                "user_email"
             ],
             [
                "user_details.user_id"
             ],
             [
                "CONCAT(first_name, ' ', last_name)",
                "user_fullname"
             ]
          ],
          includes:[
             {
                "table_name":"user_details",
                "join_type":"INNER",
                "join_condition":"users.id = user_details.user_id"
             }
          ],
          group_by:[
             "user_details.user_id",
             "last_name",
             "email",
             "first_name",
             "users.id"
          ]
       },
      },
      {
        name: 'Reject User Mail',
        slug: 'reject-user-mail',
        status_id: 1,
        query_information: {
          company_id:1,
          search_all:[
             {
                "value":"@process.unique_id",
                "operator":"=",
                "column_name":"users.id"
             }
          ],
          limit_range:1,
          print_query:false,
          start_index:0,
          sort_columns:[
             [
                "users.email",
                "asc"
             ]
          ],
          primary_table:"users",
          select_columns:[
             [
                "email",
                "user_email"
             ],
             [
                "user_details.user_id"
             ],
             [
                "CONCAT(first_name, ' ', last_name)",
                "user_fullname"
             ]
          ],
          includes:[
             {
                "table_name":"user_details",
                "join_type":"INNER",
                "join_condition":"users.id = user_details.user_id"
             }
          ],
          group_by:[
             "user_details.user_id",
             "last_name",
             "email",
             "first_name",
             "users.id"
          ]
       },
      },
    ];

    for (const process of datas) {
      await findOrCreateProcess(process, company_id);
    }

    const tags = [
      {
        description: 'It will return firstname of the mail receiving user',
        name: 'user_firstname',
        slug: '[user_firstname]',
        status_id: 1,
      },
      {
        description: 'It will return fullname of the mail receiving user',
        name: 'user_fullname',
        slug: '[user_fullname]',
        status_id: 1,
      },
      {
        description: 'It will be replaced with otp',
        name: 'otp',
        slug: '[otp]',
        status_id: 1,
      },
      {
        description: 'User Registration',
        name: 'user_email',
        slug: '[user_email]',
        status_id: 1,
      },
      {
        description: 'It will return the last name of the mail receiving user',
        name: 'user_lastname',
        slug: '[user_lastname]',
        status_id: 1,
      },
      {
        description: 'It will return the username of the mail receiving user',
        name: 'username',
        slug: '[username]',
        status_id: 1,
      },
      {
        description: "It will return the user's phone number",
        name: 'user_phone',
        slug: '[user_phone]',
        status_id: 1,
      },
      {
        description: "It will return the user's address",
        name: 'user_address',
        slug: '[user_address]',
        status_id: 1,
      },
      {
        description: "It will return the user's roles in comma separated text",
        name: 'user_roles',
        slug: '[user_roles]',
        status_id: 1,
      },
      {
        description: "Custom Email reason during approval process",
        name: 'email_custom_reason',
        slug: '[email_custom_reason]',
        status_id: 1,
      },
      {
        description: "Get manager name",
        name: 'manager_name',
        slug: '[manager_name]',
        status_id: 1,
      }
    ];

    for (const tag of tags) {
      await findOrCreateEmailTags(tag, company_id);
    }

    const emailProcessAndTagGroup = [
      {
        email_template_process: 'forgot-password',
        email_template_tag: '[otp]',
      },
      {
        email_template_process: 'email-verify',
        email_template_tag: '[otp]',
      },
      {
        email_template_process: 'login-otp',
        email_template_tag: '[otp]',
      },
      {
        email_template_process: 'forgot-password',
        email_template_tag: '[user_fullname]',
      },
      {
        email_template_process: 'email-verify',
        email_template_tag: '[user_fullname]',
      },
      {
        email_template_process: 'login-otp',
        email_template_tag: '[user_fullname]',
      },
      {
        email_template_process: 'password-changed',
        email_template_tag: '[user_fullname]',
      },
      {
        email_template_process: 'profile-updated',
        email_template_tag: '[user_firstname]',
      },
      {
        email_template_process: 'profile-updated',
        email_template_tag: '[user_fullname]',
      },
      {
        email_template_process: 'profile-updated',
        email_template_tag: '[user_email]',
      },
      {
        email_template_process: 'profile-updated',
        email_template_tag: '[user_lastname]',
      },
      {
        email_template_process: 'profile-updated',
        email_template_tag: '[username]',
      },
      {
        email_template_process: 'profile-updated',
        email_template_tag: '[user_phone]',
      },
      {
        email_template_process: 'profile-updated',
        email_template_tag: '[user_address]',
      },
      {
        email_template_process: 'profile-updated',
        email_template_tag: '[user_roles]',
      },

      {
        email_template_process: 'user-created',
        email_template_tag: '[user_firstname]',
      },
      {
        email_template_process: 'user-created',
        email_template_tag: '[user_fullname]',
      },
      {
        email_template_process: 'user-created',
        email_template_tag: '[user_email]',
      },
      {
        email_template_process: 'user-created',
        email_template_tag: '[user_lastname]',
      },
      {
        email_template_process: 'user-created',
        email_template_tag: '[username]',
      },
      {
        email_template_process: 'user-created',
        email_template_tag: '[user_phone]',
      },
      {
        email_template_process: 'user-created',
        email_template_tag: '[user_address]',
      },
      {
        email_template_process: 'user-created',
        email_template_tag: '[user_roles]',
      },
      {
        email_template_process: 'mail-import',
        email_template_tag: '[user_fullname]',
      },
      {
        email_template_process: 'mail-import',
        email_template_tag: '[user_email]',
      },
      {
        email_template_process: 'approval-user-mail',
        email_template_tag: '[user_fullname]',
      },
      {
        email_template_process: 'approval-user-mail',
        email_template_tag: '[user_email]',
      },
      {
        email_template_process: 'reject-user-mail',
        email_template_tag: '[user_fullname]',
      },
      {
        email_template_process: 'reject-user-mail',
        email_template_tag: '[user_email]',
      },
      {
        email_template_process: 'approve-timesheet-mail',
        email_template_tag: '[manager_name]',
      },
      {
        email_template_process: 'approve-timesheet-mail',
        email_template_tag: '[email_custom_reason]',
      },
      {
        email_template_process: 'reject-timesheet-mail',
        email_template_tag: '[manager_name]',
      },
      {
        email_template_process: 'reject-timesheet-mail',
        email_template_tag: '[email_custom_reason]',
      },
    ];

    for (const map of emailProcessAndTagGroup) {
      await findOrCreateEmailTagsAndTemplateMap(map, company_id);
    }

    const recipientTags = [
      {
        description: 'User who requested the api',
        slug: 'requested_user',
        name: 'requested_user',
        status_id: 1,
        query_information: {
          company_id: 1,
          search_all: [
            {
              value: '3',
              operator: '!=',
              column_name: 'users.status_id',
            },
            {
              value: '@process.user_id',
              operator: '=',
              column_name: 'users.id',
            },
          ],
          limit_range: 25,
          print_query: false,
          start_index: 0,
          sort_columns: [['users.email', 'asc']],
          primary_table: 'users',
          select_columns: [
            ['user_id'],
            ['email'],
            ['email', 'email_to'],
            ['first_name'],
            ["CONCAT(first_name, ' ', last_name)", 'full_name'],
            ['role'],
          ],
          includes: [
            {
              table_name: 'user_details',
              join_type: 'INNER',
              join_condition: 'users.id = user_details.user_id',
            },
          ],
        },
      },
      {
        description: 'Super admin of the user who requested the api',
        slug: 'super_admin',
        name: 'super_admin',
        status_id: 1,
        query_information: {
          primary_table: 'users',
          select_columns: [['b.email', 'email']],
          includes: [
            {
              table_name: 'users as b',
              join_type: 'LEFT',
              join_condition:
                "users.company_id = b.company_id AND b.role='super_admin'",
            },
          ],
          search_all: [
            {
              column_name: 'b.status_id',
              operator: '=',
              value: 1,
            },
            {
              column_name: 'users.id',
              operator: '=',
              value: '@process.user_id',
            },
          ],
          company_id: 1,
          start_index: 0,
          print_query: false,
        },
      },
      {
        description: 'Department Head of the user who requested the api',
        slug: 'department_head',
        name: 'department_head',
        status_id: 1,
        query_information: {
          primary_table: 'user_details',
          select_columns: [['users.email', 'email']],
          includes: [
            {
              table_name: 'departments as d',
              join_type: 'INNER',
              join_condition: 'user_details.department_id = d.id',
            },
            {
              table_name: 'users',
              join_type: 'INNER',
              join_condition: 'users.id = d.department_head_id',
            },
          ],
          search_all: [
            {
              column_name: 'users.status_id',
              operator: '=',
              value: '1',
            },
            {
              column_name: 'user_details.id',
              operator: '=',
              value: '@process.user_id',
            },
          ],
          company_id: 1,
          limit_range: 10,
          start_index: 0,
          print_query: false,
        },
      },
    ];

    for (const tag of recipientTags) {
      await findOrCreateEmailRecipientTags(tag, company_id);
    }

    console.log('Email Template process creation completed.');
  } catch (error) {
    throw error;
  }
}
