import { PrismaClient } from '@prisma/pg/client';

const prisma = new PrismaClient();

const barcodeTemplates = [
  {
    prefix: "BX",
    label_start: "000000",
    label_end: "000000",
    name: "boxfile barcode",
    zpl_code: `^XA
^FO30,5^GFA,1134,1134,14,gH03E,gH07C,gH0FC,gG01F8,gG01F,gG03F,gG07C,gG0F8,gG0F,g01E,g03E,g07C,V04I078,V06I0F8,V0F001F,V0F003E,V0F003C,V0F007C,V0F00F8,V0F01F,V0F3FF,V073FE,V07BFC,V03DF8,W0C6,,::::M06W01C,L03EW07C,L03EW0FC,:L03E00FT0FC,L03E01FT0FC,::L03E00ET0FC,L03EW0FC,::07FC003E007T0FC,3IF003E01FT0FC,301F803E01FI07F8I07FC00FC,I0FC03E01F001E1E001F0F00FC,I07C03E01F007C0F003C0780FC,I07C03E01F00780F807C0780FC,I07C03E01F00F807807807C0FC,I07C03E01F01F00780F807C0FC,I07C03E01F01F007C0F807C0FC,00FFC03E01F01F007C1F807C0FC,0F87C03E01F03F03FC1F83FC0FC,1E07C03E01F03JF81JFC0FC,7C07C03E01F03FJ01F8J0FC,7807C03E01F03FJ01F8J0FC,F807C03E01F01FJ01F8J0FC,:F807C03E01F01F8J0F8J0FC,F807C03E01F00F8J0FCJ0FC,F807C03E01F00FCJ07CJ0FC,7C1FC03E01F007C01007E0080FC,7FF7C03E01F003F8F001FCF80FC,3FC7C03E01FI0FFCI0FFC00FC,O01F,::::::::::O01E,O03E,:O03C,M01FF8,N07E,^FS
^FX Third section with bar code.
^FO120,120
^A0N,40,40
^FD[[label]]^FS
^FX
^CF0,40
^FO200,40^FDBox File^FS
^XZ`,
    created_by: 1,
    page_width: 2,
    page_height: 1
  },
  {
    prefix: "CNT",
    label_start: "000000",
    label_end: "000000",
    name: "carton barcode",
    zpl_code: `^XA
^FO5,0^GFA,6402,6402,33,hU07C,hT0IF,hS01IF,hS03FFE,hS07FFC,hS07FF8,hS0IF8,hR01IF,hR03FFE,:hR07FFC,hR0IF8,hQ01IF8,hQ01IF,hQ03FFE01F8,hQ07FF80C02,hQ0IF02001,hQ0FFE04I08,hP01FFC08I04,hP03FF82J04,hP03FF04,hP07FE08J02,hP0FFC1J07E,hO01FFC2I06,hO01FF84I08,hO03FF08002,hO07FE1I08,hO0FFE2001,hO0FFCI02,hN01FF84,hN03FF88004,hN07FF1I08,hN07FEI01,hN0FFC2,hM01FFC4002,hM01FF8I04,hM03FF08008,hM07FF10018,hM0FFEI01,hM0FFC2002,hL01FF84004,hL03FF88,hL07FFJ08,hL07FE1001,hL0FFE2,hK03FFC4002,hK07FF84004,hJ01IF88008,hJ07IF1,hH0KFEI01,hH0KFC2002,hH07JFC4004,hH07JF8I04,hH03JF08008,hH03JF1001,hH01IFCI02,hI0IF82,hI07FE04004,hJ0E008,,hL01001,hL02002,,hL04,hL08008,hN01,hK01002,hK02,hK04004,hN08,hK0802,hK0FF8,,W07EhI0FC,V03FEhH07FC,V0FFEhG01FFC,U01FFEhG03FFC,::::U01FFEK07EgT03FFC,U01FFEK0FF8gS03FFC,U01FFEJ01FF8gS03FFC,U01FFEJ01FFCgS03FFC,U01FFEJ03FFCgS03FFC,::::U01FFEJ01FFCgS03FFC,U01FFEK0FF8gS03FFC,U01FFEK07FgT03FFC,U01FFEhG03FFC,:::::::I01JF8L01FFEL03CgS03FFC,001LFL01FFEK03FCgS03FFC,00MFCK01FFEJ01FFCgS03FFC,03FF87IFEK01FFEJ03FFCgS03FFC,0FEI03IFK01FFEJ03FFCO078S0F8M03FFC,0F8J0IF8J01FFEJ03FFCM03IFEP07IFCL03FFC,04K07FFCJ01FFEJ03FFCL01FF8FFCN01FF0FF8K03FFC,M07FFCJ01FFEJ03FFCL07FC01FFN0FFC01FEK03FFC,M03FFEJ01FFEJ03FFCK01FF8007F8L01FFI0FF8J03FFC,M03FFEJ01FFEJ03FFCK03FEI03FCL07FEI07FCJ03FFC,M01FFEJ01FFEJ03FFCK07FEI03FEL0FFCI03FEJ03FFC,M01FFEJ01FFEJ03FFCK0FFCI01FFK01FF8I03FFJ03FFC,M01FFEJ01FFEJ03FFCJ01FF8I01FF8J03FF8I01FFJ03FFC,M01FFEJ01FFEJ03FFCJ03FF8J0FF8J03FFJ01FF8I03FFC,N0IFJ01FFEJ03FFCJ03FFK0FFCJ07FFJ01FF8I03FFC,N0IFJ01FFEJ03FFCJ07FFK0FFCJ0FFEJ01FFCI03FFC,N0IFJ01FFEJ03FFCJ07FFK0FFEJ0FFEJ01FFCI03FFC,N0IFJ01FFEJ03FFCJ0FFEK0FFEI01FFEK0FFCI03FFC,N0IFJ01FFEJ03FFCJ0FFEK0FFEI01FFEK0FFEI03FFC,M01IFJ01FFEJ03FFCI01FFEK0FFEI01FFEK0FFEI03FFC,M01IFJ01FFEJ03FFCI01FFEK0FFEI03FFCK0FFEI03FFC,:K0LFJ01FFEJ03FFCI03FFEK0IFI03FFCK0FFEI03FFC,I01IF1IFJ01FFEJ03FFCI03FFEK0IFI03FFCK0FFEI03FFC,I0FF800IFJ01FFEJ03FFCI03FFEK0FFEI07FFCJ01FFEI03FFC,007FCI0IFJ01FFEJ03FFCI03PFEI07PFEI03FFC,00FFJ0IFJ01FFEJ03FFCI03PFEI07PFEI03FFC,03FEJ0IFJ01FFEJ03FFCI03FFEQ07FFCQ03FFC,07FCJ0IFJ01FFEJ03FFCI07FFEQ07FFCQ03FFC,0FF8J0IFJ01FFEJ03FFCI07FFEQ07FFCQ03FFC,1FF8J0IFJ01FFEJ03FFCI03FFEQ07FFCQ03FFC,3FFK0IFJ01FFEJ03FFCI03FFEQ07FFCQ03FFC,:7FFK0IFJ01FFEJ03FFCI03FFEQ07FFCQ03FFC,7FFK0IFJ01FFEJ03FFCI03FFEQ07FFEQ03FFC,7FEK0IFJ01FFEJ03FFCI03FFEQ03FFEQ03FFC,7FEK0IFJ01FFEJ03FFCI01FFEQ03FFEQ03FFC,7FEK0IFJ01FFEJ03FFCI01IFQ03FFEQ03FFC,FFEK0IFJ01FFEJ03FFCI01IFQ01FFEQ03FFC,FFEK0IFJ01FFEJ03FFCI01IFQ01IFQ03FFC,FFEK0IFJ01FFEJ03FFCJ0IF8P01IFQ03FFC,7FFK0IFJ01FFEJ03FFCJ0IF8Q0IFQ03FFC,7FFK0IFJ01FFEJ03FFCJ07FF8Q0IF8P03FFC,7FFJ01IFJ01FFEJ03FFCJ07FFCQ07FF8P03FFC,7FF8I03IFJ01FFEJ03FFCJ03FFEQ07FFCP03FFC,7FF8I07IFJ01FFEJ03FFCJ01FFEQ03FFEP03FFC,3FFC001CIFJ01FFEJ03FFCK0IF8J038J01IFK07J03FFC,3IF007CIFJ01FFEJ03FFCK0IFCJ0F8K0IF8I01FJ03FFC,1LF0IFJ01FFEJ03FFCK03IFI0FEL07IFI0FEJ03FFC,0KFE0IFJ01FFEJ03FFCK01MF8L01MF8J03FFC,0KFC0IFJ01FFEJ03FFCL07KFEN07KFEK03FFC,03JF00IFJ01FFEJ03FFCM0KFO01KFL03FFC,00IFI0IFJ01FFEJ03FFCN0IFR0FFEM03FFC,gI03FFC,::::::::::::::::::::gI03FF8,::::gI07FF,:::gI07FE,gI0FFE,gI0FFC,gH01FF8,gH01FF,Y04007FF,Y0F81FFC,Y0KF8,Y01IFC,gG0F8,^FS
^FX Third section with bar code.
^FO60,200
^BQN,2,10
^FDLA,[[label]]^FS
^FO350,300
^A0N,50,50
^FD[[label]]^FS
^FX
^CF0,80
^FO300,75^FDCarton Box^FS
^CF0,50
^FO60,475^FDFinance_Dept_AR2022^FS
^XZ`,
    created_by: 1,
    page_width: 4,
    page_height: 3
  },
];

export async function barcodeTemplateSeeder(company_id: number) {
  try {
    // Fetch the super admin user for this company
    let user = await prisma.user.findFirst({
      where: { company_id: company_id, role: 'super_admin' },
    });

    if (!user) {
      console.log(`Barcode template seeder failed: no admin user available for company id ${company_id}`);
      return false;
    }

    const createdBy = user.id;

    for (const template of barcodeTemplates) {
      // Fetch the last created label_end for this template prefix if it exists
      const lastTemplate = await prisma.barcodeTemplates.findFirst({
        where: { prefix: template.prefix, company_id: company_id },
        orderBy: { created_at: 'desc' }
      });

      // Set label_start based on the last label_end if available, otherwise use template's default
      const labelStart = lastTemplate?.label_end || template.label_start;

      const existingTemplate = await prisma.barcodeTemplates.findFirst({
        where: { prefix: template.prefix, company_id: company_id },
      });

      if (!existingTemplate) {
        await prisma.barcodeTemplates.create({
          data: {
            company_id: company_id,
            prefix: template.prefix,
            label_start: lastTemplate?.label_start || template.label_start,
            label_end: lastTemplate?.label_end || template.label_start,  // Assuming label_end should start as template's default
            name: template.name,
            zpl_code: template.zpl_code,
            page_width: template.page_width,
            page_height: template.page_height,
            status_id: 1,
            created_by: createdBy,
          },
        });
        console.log(`Created barcode template with prefix ${template.prefix}: ${template.name}`);
      } else {
        console.log(`Barcode template with prefix ${template.prefix} already exists.`);
      }
    }
  } catch (error) {
    console.error(`Error in barcodeTemplateSeeder: ${error}`);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
