import { PrismaClient } from '@prisma/sql/client';

const prisma = new PrismaClient();

interface EmailTemplateData {
  email_template_process_slug: string;
  template_slug?: string;
  recipient_type?: "tag" | "user_id" | "custom";
  email_to?: string;
}


export async function emailTemplateAssignmentsSeeder(company_id: number) {
  try {
    console.log("🚀 Starting Email Template Assignments Seeder...");

    const user = await prisma.user.findFirst({
      where: { company_id, role: 'super_admin' },
    });

    if (!user) {
      console.log(`❌ Email Templates Assignment Seeder failed: No super_admin user found for company_id ${company_id}`);
      return false;
    }

    const created_by = user.id;
    
    const datas:EmailTemplateData[] =[
      {
        "email_template_process_slug": "user-created",
        "template_slug": "user_template_user_created",
        "recipient_type": "tag",
        "email_to": "requested_user"
      },
      {
        "email_template_process_slug": "user-created",
        "template_slug": "admin_template_user_created",
        "recipient_type": "tag",
        "email_to": "super_admin"
      },
      {
        "email_template_process_slug": "user-created",
        "template_slug": "department_head_template_user_created",
        "recipient_type": "tag",
        "email_to": "department_head"
      },
      {
        "email_template_process_slug": "forgot-password",
        "template_slug": "forgot-password",
        "recipient_type": "tag",
        "email_to": "requested_user"
      },
      {
        "email_template_process_slug": "login-otp",
        "template_slug": "login-otp",
        "recipient_type": "tag",
        "email_to": "requested_user"
      },
      {
        "email_template_process_slug": "email-verify",
        "template_slug": "email-verify",
        "recipient_type": "tag",
        "email_to": "requested_user"
      }
    ];
    
    const assignmentsToCreate:any = [];

    for (const data of datas) {
      const emailTemplateProcess = await prisma.emailTemplateProcess.findFirst({
        where: { slug: data.email_template_process_slug, company_id },
      });

      if (!emailTemplateProcess) {
        console.log(`⚠️ Skipping: Email Template Process with slug "${data.email_template_process_slug}" not found for company_id ${company_id}.`);
        continue;
      }

      let template_id: number | null = null;
      if (data.template_slug) {
        const template = await prisma.emailTemplate.findFirst({
          where: { slug: data.template_slug, company_id },
        });

        if (template) {
          template_id = template.id;
        } else {
          console.error(`❌ Skipping: Template with slug "${data.template_slug}" not found for company_id ${company_id}.`);
          continue;
        }
      }

      let email_to: string | null = data.email_to || null;
      if (data.recipient_type === "tag" && data.email_to) {
        const template = await prisma.emailTemplateRecipientTag.findFirst({
          where: { slug: data.email_to, company_id },
        });

        if (template) {
          email_to = `${template.id}`;
        } else {
          console.error(`❌ Skipping: Email template recipient tag with slug "${data.email_to}" not found for company_id ${company_id}.`);
          continue;
        }
      }

      const email_template_process_id = emailTemplateProcess.id;

      const existingAssignment = await prisma.emailTemplateAssignment.findFirst({
        where: {
          email_template_process_id,
          template_id: template_id || 0,
          recipient_type: data.recipient_type,
          email_to,
          company_id,
        },
      });

      if (!existingAssignment && template_id) {
        assignmentsToCreate.push(
          prisma.emailTemplateAssignment.create({
            data: {
              email_template_process_id,
              template_id,
              recipient_type: data.recipient_type!,
              email_to,
              company_id,
              created_by,
              created_at: new Date(),
            },
          }).then(() => {
            console.log(`✅ Created Email Template Assignment for company_id ${company_id}.`);
          }).catch(error => {
            console.error(`❌ Error creating email template assignment:`, error);
          })
        );
      } else {
        console.log(`🔄 Email Template Assignment already exists for company_id ${company_id}.`);
      }
    }

    await Promise.all(assignmentsToCreate);

    console.log("✅ Email Templates Assignment Seeder completed successfully.");
  } catch (error) {
    console.error(`❌ Error in emailTemplateAssignmentsSeeder: ${error}`);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
