import { PrismaClient } from '@prisma/sql/client';
import fs from 'fs';
import path from 'path';

const prisma = new PrismaClient();

// Load push and pull data transfer queries from JSON files
const pushDataTransferSeed = JSON.parse(
  fs.readFileSync(path.join(__dirname, 'json/data-transfer-queries-push.json'), 'utf-8')
);
const pullDataTransferSeed = JSON.parse(
  fs.readFileSync(path.join(__dirname, 'json/data-transfer-queries-pull.json'), 'utf-8')
);

export async function dataTransferQueriesSeeder(company_id: number) {
  try {
    console.log("🚀 Starting Data Transfer Queries Seeder...");

    // Fetch a super admin user for the given company
    const user = await prisma.user.findFirst({
      where: { company_id, role: 'super_admin' },
    });

    if (!user) {
      console.log(`❌ Data Transfer Queries Seeder failed: No super_admin user found for company_id ${company_id}`);
      return false;
    }

    const createdBy = user.id;
    const utcNow = new Date();

    const allData = [...pushDataTransferSeed, ...pullDataTransferSeed];

    const queriesToUpsert = allData.map(item =>
      prisma.dataTransferQueries.upsert({
        where: { company_id_transfer_type_job_name: {
          company_id,
          transfer_type:item.transfer_type,
          job_name:item.job_name
        }}, 
        update: {},
        create: {
          ...item,
          company_id,
          created_by: createdBy,
          created_at: utcNow,
        },
      }).then(() => {
        console.log(`✅ Upserted Data Transfer Query ID ${item.job_name} for company_id ${company_id}`);
      }).catch(error => {
        console.error(`❌ Error upserting Data Transfer Query ID ${item.job_name}:`, error);
      })
    );

    await Promise.all(queriesToUpsert);

    console.log("✅ Data Transfer Queries Seeder completed successfully.");
  } catch (error) {
    console.error(`❌ Error in dataTransferQueriesSeeder: ${error}`);
    throw error;
  } finally {
    await prisma.$disconnect();
  }
}
