import { PrismaClient } from '@prisma/pg/client';
import { v4 as uuidv4 } from 'uuid';

const prisma = new PrismaClient();

function getCurrentUtcTime() {
    return new Date().toISOString();
}

export async function roleSeeds(company_id: number) {
    const roles = ['Organization Admin'];
    try {
        let user = await prisma.user.findFirst({
            where: { company_id: company_id, role: 'super_admin' }
        });
        if (!user) {
            console.log(`Role seeder failed: admin user not available for company id ${company_id}`);
            return false;
        }

        const createdBy = user.id;
        const utcNow = getCurrentUtcTime();

        // Fetch existing roles for the company
        const existingRoles = await prisma.role.findMany({
            where: { company_id, name: { in: roles } }
        });

        const existingRoleNames = new Set(existingRoles.map(role => role.name));

        const rolesPromises: Promise<void>[] = [];

        for (let roleName of roles) {
            if (!existingRoleNames.has(roleName)) {
                rolesPromises.push(
                    prisma.role.create({
                        data: {
                            uuid: uuidv4(),
                            name: roleName,
                            company_id,
                            created_by: createdBy,
                            created_at: utcNow,
                        },
                    }).then(role => {
                        console.log(`Role ${role.name} created successfully. id is ${role.id}`);
                    }).catch(error => {
                        console.error(`Failed to create role ${roleName} for user_id ${createdBy}:`, error);
                    })
                );
            } else {
                console.log(`Role ${roleName} already exists for company_id ${company_id}, skipping creation.`);
            }
        }

        await Promise.all(rolesPromises);

        console.log('Role seeding completed.');

    } catch (error) {
        console.error('Role seeding failed:', error);
    } finally {
        // await prisma.$disconnect();
    }
}
