CREATE OR REPLACE PROCEDURE get_configurations_values_v1(
    IN request JSONB,
    INOUT result JSONB
)
LANGUAGE plpgsql AS $$
DECLARE
    rec RECORD;
    current_category_id TEXT;
    user_rec RECORD;
    config_map JSONB := '{}'::JSONB;
    categories JSONB;
    input_user_id INT;
    latest_updated TIMESTAMP := NULL;
    temp_updated TIMESTAMP;
BEGIN
    -- Initialize the result as an empty JSON object if it's not provided
    IF result IS NULL THEN
        result := '{}'::JSONB;
    END IF;

    -- Extract categories and user_id from the request JSONB
    categories := (request->>'categories')::JSONB;

    -- Safely extract user_id, handle null case
    IF request ? 'user_id' AND request->>'user_id' != '' AND request->>'user_id' != 'null' THEN
        BEGIN
            input_user_id := (request->>'user_id')::INT;
        EXCEPTION 
            WHEN others THEN
                input_user_id := NULL;
        END;
    ELSE
        input_user_id := NULL;
    END IF;

    -- Check if categories exists
    IF categories IS NULL THEN
        result := '{"error": "categories parameter is required"}'::JSONB;
        RETURN;
    END IF;

    -- Loop through each value in the categories object
    FOR current_category_id IN 
        SELECT jsonb_each_text.value 
        FROM jsonb_each_text(categories)
    LOOP
        -- 1. Fetch all default configs for this category
        FOR rec IN 
            SELECT ac.config_key, ac.config_value, ac.updated_at
            FROM app_configurations ac
            WHERE ac.category_id = current_category_id 
            ORDER BY ac.order_no ASC
        LOOP
            config_map := jsonb_set(
                config_map,
                ARRAY[rec.config_key],
                COALESCE(to_jsonb(rec.config_value), 'null'::JSONB),
                TRUE
            );
            
            -- Track latest updated_at
            IF rec.updated_at IS NOT NULL THEN
                IF latest_updated IS NULL OR rec.updated_at > latest_updated THEN
                    latest_updated := rec.updated_at;
                END IF;
            END IF;
        END LOOP;

        -- 2. If user_id is provided, fetch user configs and override/add
        IF input_user_id IS NOT NULL THEN
            FOR user_rec IN 
                SELECT auc.config_key, auc.config_value, auc.updated_at
                FROM app_user_configurations auc
                WHERE auc.category_id = current_category_id
                  AND auc.user_id = input_user_id
            LOOP
                config_map := jsonb_set(
                    config_map,
                    ARRAY[user_rec.config_key],
                    COALESCE(to_jsonb(user_rec.config_value), 'null'::JSONB),
                    TRUE
                );
                
                -- Track latest updated_at
                IF user_rec.updated_at IS NOT NULL THEN
                    IF latest_updated IS NULL OR user_rec.updated_at > latest_updated THEN
                        latest_updated := user_rec.updated_at;
                    END IF;
                END IF;
            END LOOP;
        END IF;
    END LOOP;

    -- Final result
    result := jsonb_set(result, '{data}', config_map, TRUE);
    IF latest_updated IS NOT NULL THEN
        result := jsonb_set(result, '{latest_updated_at}', to_jsonb(latest_updated::TEXT), TRUE);
    END IF;
END;
$$;




CREATE OR REPLACE PROCEDURE get_configurations_values_v2(
    IN request JSONB,
    INOUT result JSONB
)
LANGUAGE plpgsql AS $$
DECLARE
    rec RECORD;
    current_category_id TEXT;
    user_rec RECORD;
    config_map JSONB := '{}'::JSONB;
    categories JSONB;
    input_org_code TEXT;
    input_user_id INT;
    organization_id INT;
    latest_updated TIMESTAMP := NULL;
    temp_updated TIMESTAMP;
BEGIN
    -- Initialize the result as an empty JSON object if it's not provided
    IF result IS NULL THEN
        result := '{}'::JSONB;
    END IF;

    -- Extract categories and org_code from the request JSONB
    categories := (request->>'categories')::JSONB;
    input_org_code := request->>'org_code';  -- Get the org_code

    -- If org_code is not provided, return an error
    IF input_org_code IS NULL OR input_org_code = '' THEN
        result := '{"error": "org_code parameter is required"}'::JSONB;
        RETURN;
    END IF;

    -- Fetch the organization_id using the org_code
    SELECT o.id INTO organization_id
    FROM organizations o
    WHERE o.code = input_org_code
    LIMIT 1;

    -- If no organization found, return an error
    IF organization_id IS NULL THEN
        result := '{"error": "Invalid org_code"}'::JSONB;
        RETURN;
    END IF;

    -- Get user_id from user_details using the organization_id
    SELECT u.user_id INTO input_user_id
    FROM user_details u
    WHERE u.organization_id = organization_id
    LIMIT 1;

    -- If no user found for the organization, return an error
    IF input_user_id IS NULL THEN
        result := '{"error": "No user found for the provided org_code"}'::JSONB;
        RETURN;
    END IF;

    -- Check if categories exists
    IF categories IS NULL THEN
        result := '{"error": "categories parameter is required"}'::JSONB;
        RETURN;
    END IF;

    -- Loop through each value in the categories object
    FOR current_category_id IN 
        SELECT jsonb_each_text.value 
        FROM jsonb_each_text(categories)
    LOOP
        -- 1. Fetch all default configs for this category
        FOR rec IN 
            SELECT ac.config_key, ac.config_value, ac.updated_at
            FROM app_configurations ac
            WHERE ac.category_id = current_category_id 
            ORDER BY ac.order_no ASC
        LOOP
            config_map := jsonb_set(
                config_map,
                ARRAY[rec.config_key],
                COALESCE(to_jsonb(rec.config_value), 'null'::JSONB),
                TRUE
            );
            
            -- Track latest updated_at
            IF rec.updated_at IS NOT NULL THEN
                IF latest_updated IS NULL OR rec.updated_at > latest_updated THEN
                    latest_updated := rec.updated_at;
                END IF;
            END IF;
        END LOOP;

        -- 2. If user_id is provided, fetch user configs and override/add
        IF input_user_id IS NOT NULL THEN
            FOR user_rec IN 
                SELECT auc.config_key, auc.config_value, auc.updated_at
                FROM app_user_configurations auc
                WHERE auc.category_id = current_category_id
                  AND auc.user_id = input_user_id
            LOOP
                config_map := jsonb_set(
                    config_map,
                    ARRAY[user_rec.config_key],
                    COALESCE(to_jsonb(user_rec.config_value), 'null'::JSONB),
                    TRUE
                );
                
                -- Track latest updated_at
                IF user_rec.updated_at IS NOT NULL THEN
                    IF latest_updated IS NULL OR user_rec.updated_at > latest_updated THEN
                        latest_updated := user_rec.updated_at;
                    END IF;
                END IF;
            END LOOP;
        END IF;
    END LOOP;

    -- Final result
    result := jsonb_set(result, '{data}', config_map, TRUE);
    IF latest_updated IS NOT NULL THEN
        result := jsonb_set(result, '{latest_updated_at}', to_jsonb(latest_updated::TEXT), TRUE);
    END IF;
END;
$$;



CREATE OR REPLACE PROCEDURE compare_last_updated_at(
    IN request JSONB,
    INOUT result JSONB
)
LANGUAGE plpgsql AS $$
DECLARE
    input_org_code TEXT;
    input_last_updated_at TIMESTAMP;
    org_id INT;  -- Renamed to avoid ambiguity
    input_user_id INT;
    user_config_last_updated TIMESTAMP;
BEGIN
    -- Extract the org_code and last_updated_at from the request JSONB
    input_org_code := request->>'org_code';
    input_last_updated_at := (request->>'last_updated_at')::TIMESTAMP;

    -- If org_code or last_updated_at is not provided, return an error
    IF input_org_code IS NULL OR input_last_updated_at IS NULL THEN
        result := '{"error": "org_code and last_updated_at parameters are required"}'::JSONB;
        RETURN;
    END IF;

    -- Fetch the org_id using the org_code
    SELECT o.id INTO org_id
    FROM organizations o
    WHERE o.code = input_org_code
    LIMIT 1;

    -- If no organization found, return an error
    IF org_id IS NULL THEN
        result := '{"error": "Invalid org_code"}'::JSONB;
        RETURN;
    END IF;

    -- Get the user_id from user_details using the org_id
    SELECT u.user_id INTO input_user_id
    FROM user_details u
    WHERE u.organization_id = org_id  -- Ensure the use of alias 'u.organization_id' to avoid ambiguity
    LIMIT 1;

    -- If no user found for the organization, return an error
    IF input_user_id IS NULL THEN
        result := '{"error": "No user found for the provided org_code"}'::JSONB;
        RETURN;
    END IF;

    -- Get the latest updated_at value from app_user_configurations for the user_id
    SELECT auc.updated_at INTO user_config_last_updated
    FROM app_user_configurations auc
    WHERE auc.user_id = input_user_id
    ORDER BY auc.updated_at DESC  -- Order by latest updated_at
    LIMIT 1;  -- Get the row with the latest updated_at

    -- If no last_updated_at value found, return an error
    IF user_config_last_updated IS NULL THEN
        result := '{"error": "No configuration found for the user"}'::JSONB;
        RETURN;
    END IF;

    -- Compare the provided last_updated_at with the value from the table
    IF input_last_updated_at < user_config_last_updated THEN
        result := '{"result": true}'::JSONB;
    ELSE
        result := '{"result": false}'::JSONB;
    END IF;
    
END;
$$;
